;-----------------------------------------------------------------------------;
; Fixed-point FFT routines for megaAVRs                        (C)ChaN, 2005
;-----------------------------------------------------------------------------;
;
; void fft_input (const int16_t *array_src, complex_t *array_bfly);
; void fft_execute (complex_t *array_bfly);
; void fft_output (complex_t *array_bfly, uint16_t *array_dst);
;
;  <array_src>: Wave form to be processed.
;  <array_bfly>: Complex array for butterfly operations.
;  <array_dst>: Spectrum output buffer.
;
; These functions must be called in sequence to do a DFT in FFT algorithm.
; fft_input() fills the complex array with a wave form to prepare butterfly
; operations. A hamming window is applied at the same time.
; fft_execute() executes the butterfly operations.
; fft_output() re-orders the results, converts the complex spectrum into
; scalar spectrum and output it in linear scale.
;
; The number of points FFT_N is defined in "ffft.h" and the value can be
; power of 2 in range of 64 - 1024.
;
;----------------------------------------------------------------------------;
; 16bit fixed-point FFT performance with MegaAVRs
; (Running at 16MHz/internal SRAM)
;
;  Points:   Input, Execute,  Output,    Total:  Throughput
;   64pts:   .17ms,   2.0ms,   1.2ms,    3.4ms:   19.0kpps
;  128pts:   .33ms,   4.6ms,   2.4ms,    7.3ms:   17.5kpps
;  256pts:   .66ms,  10.4ms,   4.9ms,   15.9ms:   16.1kpps
;  512pts:   1.3ms,  23.2ms,   9.7ms,   34.2ms:   14.9kpps
; 1024pts:   2.7ms,  51.7ms,  19.4ms,   73.7ms:   13.9kpps
;----------------------------------------------------------------------------;


.nolist
#define FFFT_ASM
#include "ffft.h"
.list

#define FFT_B_512 9

;----------------------------------------------------------------------------;
; Constant Tables

.global tbl_window_512
tbl_window_512:	; tbl_window[] = ... (This is a Hamming window)
#if FFT_N_512 == 512
	.dc.w	2621, 2622, 2625, 2631, 2639, 2649, 2662, 2676, 2693, 2713, 2734, 2758, 2784, 2812, 2843, 2876
	.dc.w	2910, 2948, 2987, 3029, 3073, 3119, 3167, 3217, 3270, 3325, 3382, 3441, 3502, 3565, 3631, 3698
	.dc.w	3768, 3840, 3914, 3990, 4068, 4148, 4230, 4314, 4401, 4489, 4579, 4671, 4765, 4861, 4959, 5059
	.dc.w	5161, 5265, 5370, 5478, 5587, 5698, 5811, 5926, 6042, 6160, 6280, 6402, 6525, 6651, 6777, 6906
	.dc.w	7036, 7167, 7300, 7435, 7571, 7709, 7849, 7989, 8132, 8275, 8420, 8567, 8715, 8864, 9015, 9166
	.dc.w	9320, 9474, 9630, 9787, 9945, 10104, 10264, 10426, 10588, 10752, 10917, 11082, 11249, 11417, 11586, 11755
	.dc.w	11926, 12097, 12269, 12442, 12616, 12790, 12966, 13142, 13318, 13496, 13674, 13852, 14031, 14211, 14391, 14572
	.dc.w	14753, 14935, 15117, 15299, 15482, 15665, 15849, 16032, 16216, 16400, 16585, 16769, 16954, 17139, 17324, 17509
	.dc.w	17694, 17879, 18064, 18248, 18433, 18618, 18802, 18987, 19171, 19355, 19539, 19722, 19905, 20088, 20271, 20453
	.dc.w	20634, 20815, 20996, 21176, 21356, 21535, 21714, 21892, 22069, 22246, 22422, 22597, 22772, 22945, 23118, 23290
	.dc.w	23462, 23632, 23802, 23970, 24138, 24305, 24471, 24635, 24799, 24962, 25123, 25283, 25443, 25601, 25758, 25913
	.dc.w	26068, 26221, 26373, 26523, 26673, 26820, 26967, 27112, 27256, 27398, 27539, 27678, 27816, 27952, 28087, 28220
	.dc.w	28352, 28482, 28610, 28737, 28862, 28985, 29107, 29227, 29345, 29462, 29576, 29689, 29800, 29910, 30017, 30123
	.dc.w	30226, 30328, 30428, 30526, 30622, 30716, 30808, 30899, 30987, 31073, 31157, 31239, 31319, 31397, 31473, 31547
	.dc.w	31619, 31689, 31756, 31822, 31885, 31947, 32006, 32063, 32117, 32170, 32220, 32269, 32315, 32359, 32400, 32440
	.dc.w	32477, 32512, 32545, 32575, 32603, 32629, 32653, 32675, 32694, 32711, 32726, 32738, 32748, 32756, 32762, 32765
	.dc.w	32766, 32765, 32762, 32756, 32748, 32738, 32726, 32711, 32694, 32675, 32653, 32629, 32603, 32575, 32545, 32512
	.dc.w	32477, 32440, 32400, 32359, 32315, 32269, 32221, 32170, 32117, 32063, 32006, 31947, 31885, 31822, 31757, 31689
	.dc.w	31619, 31547, 31473, 31397, 31319, 31239, 31157, 31073, 30987, 30899, 30808, 30716, 30622, 30526, 30428, 30328
	.dc.w	30226, 30123, 30017, 29910, 29800, 29689, 29576, 29462, 29345, 29227, 29107, 28985, 28862, 28737, 28610, 28482
	.dc.w	28352, 28220, 28087, 27952, 27816, 27678, 27539, 27398, 27256, 27112, 26967, 26820, 26673, 26523, 26373, 26221
	.dc.w	26068, 25913, 25758, 25601, 25443, 25283, 25123, 24962, 24799, 24635, 24471, 24305, 24138, 23970, 23802, 23632
	.dc.w	23462, 23291, 23118, 22945, 22772, 22597, 22422, 22246, 22069, 21892, 21714, 21535, 21356, 21176, 20996, 20815
	.dc.w	20634, 20453, 20271, 20088, 19905, 19722, 19539, 19355, 19171, 18987, 18803, 18618, 18433, 18249, 18064, 17879
	.dc.w	17694, 17509, 17324, 17139, 16954, 16769, 16585, 16401, 16216, 16032, 15849, 15665, 15482, 15299, 15117, 14935
	.dc.w	14753, 14572, 14391, 14211, 14031, 13852, 13674, 13496, 13318, 13142, 12966, 12790, 12616, 12442, 12269, 12097
	.dc.w	11926, 11755, 11586, 11417, 11249, 11083, 10917, 10752, 10588, 10426, 10264, 10104, 9945, 9787, 9630, 9474
	.dc.w	9320, 9167, 9015, 8864, 8715, 8567, 8420, 8275, 8132, 7989, 7849, 7709, 7571, 7435, 7300, 7167
	.dc.w	7036, 6906, 6777, 6651, 6526, 6402, 6280, 6160, 6042, 5926, 5811, 5698, 5587, 5478, 5370, 5265
	.dc.w	5161, 5059, 4959, 4861, 4765, 4671, 4579, 4489, 4401, 4314, 4230, 4148, 4068, 3990, 3914, 3840
	.dc.w	3768, 3699, 3631, 3565, 3502, 3441, 3382, 3325, 3270, 3217, 3167, 3119, 3073, 3029, 2987, 2948
	.dc.w	2910, 2876, 2843, 2812, 2784, 2758, 2734, 2713, 2693, 2676, 2662, 2649, 2639, 2631, 2625, 2622
#endif


tbl_cos_sin:	; Table of {cos(x),sin(x)}, (0 <= x < pi, in FFT_N/2 steps)
#if FFT_N_512 == 512
	.dc.w	32767, 0, 32764, 402, 32757, 804, 32744, 1206, 32727, 1607, 32705, 2009, 32678, 2410, 32646, 2811
	.dc.w	32609, 3211, 32567, 3611, 32520, 4011, 32468, 4409, 32412, 4807, 32350, 5205, 32284, 5601, 32213, 5997
	.dc.w	32137, 6392, 32056, 6786, 31970, 7179, 31880, 7571, 31785, 7961, 31684, 8351, 31580, 8739, 31470, 9126
	.dc.w	31356, 9511, 31236, 9895, 31113, 10278, 30984, 10659, 30851, 11038, 30713, 11416, 30571, 11792, 30424, 12166
	.dc.w	30272, 12539, 30116, 12909, 29955, 13278, 29790, 13645, 29621, 14009, 29446, 14372, 29268, 14732, 29085, 15090
	.dc.w	28897, 15446, 28706, 15799, 28510, 16150, 28309, 16499, 28105, 16845, 27896, 17189, 27683, 17530, 27466, 17868
	.dc.w	27244, 18204, 27019, 18537, 26789, 18867, 26556, 19194, 26318, 19519, 26077, 19840, 25831, 20159, 25582, 20474
	.dc.w	25329, 20787, 25072, 21096, 24811, 21402, 24546, 21705, 24278, 22004, 24006, 22301, 23731, 22594, 23452, 22883
	.dc.w	23169, 23169, 22883, 23452, 22594, 23731, 22301, 24006, 22004, 24278, 21705, 24546, 21402, 24811, 21096, 25072
	.dc.w	20787, 25329, 20474, 25582, 20159, 25831, 19840, 26077, 19519, 26318, 19194, 26556, 18867, 26789, 18537, 27019
	.dc.w	18204, 27244, 17868, 27466, 17530, 27683, 17189, 27896, 16845, 28105, 16499, 28309, 16150, 28510, 15799, 28706
	.dc.w	15446, 28897, 15090, 29085, 14732, 29268, 14372, 29446, 14009, 29621, 13645, 29790, 13278, 29955, 12909, 30116
	.dc.w	12539, 30272, 12166, 30424, 11792, 30571, 11416, 30713, 11038, 30851, 10659, 30984, 10278, 31113, 9895, 31236
	.dc.w	9511, 31356, 9126, 31470, 8739, 31580, 8351, 31684, 7961, 31785, 7571, 31880, 7179, 31970, 6786, 32056
	.dc.w	6392, 32137, 5997, 32213, 5601, 32284, 5205, 32350, 4807, 32412, 4409, 32468, 4011, 32520, 3611, 32567
	.dc.w	3211, 32609, 2811, 32646, 2410, 32678, 2009, 32705, 1607, 32727, 1206, 32744, 804, 32757, 402, 32764
	.dc.w	0, 32766, -402, 32764, -804, 32757, -1206, 32744, -1607, 32727, -2009, 32705, -2410, 32678, -2811, 32646
	.dc.w	-3211, 32609, -3611, 32567, -4010, 32520, -4409, 32468, -4807, 32412, -5205, 32350, -5601, 32284, -5997, 32213
	.dc.w	-6392, 32137, -6786, 32056, -7179, 31970, -7571, 31880, -7961, 31785, -8351, 31684, -8739, 31580, -9126, 31470
	.dc.w	-9511, 31356, -9895, 31236, -10278, 31113, -10659, 30984, -11038, 30851, -11416, 30713, -11792, 30571, -12166, 30424
	.dc.w	-12539, 30272, -12909, 30116, -13278, 29955, -13645, 29790, -14009, 29621, -14372, 29446, -14732, 29268, -15090, 29085
	.dc.w	-15446, 28897, -15799, 28706, -16150, 28510, -16499, 28309, -16845, 28105, -17189, 27896, -17530, 27683, -17868, 27466
	.dc.w	-18204, 27244, -18537, 27019, -18867, 26789, -19194, 26556, -19519, 26318, -19840, 26077, -20159, 25831, -20474, 25582
	.dc.w	-20787, 25329, -21096, 25072, -21402, 24811, -21705, 24546, -22004, 24278, -22301, 24006, -22594, 23731, -22883, 23452
	.dc.w	-23169, 23169, -23452, 22883, -23731, 22594, -24006, 22301, -24278, 22005, -24546, 21705, -24811, 21402, -25072, 21096
	.dc.w	-25329, 20787, -25582, 20474, -25831, 20159, -26077, 19840, -26318, 19519, -26556, 19194, -26789, 18867, -27019, 18537
	.dc.w	-27244, 18204, -27466, 17868, -27683, 17530, -27896, 17189, -28105, 16845, -28309, 16499, -28510, 16150, -28706, 15799
	.dc.w	-28897, 15446, -29085, 15090, -29268, 14732, -29446, 14372, -29620, 14009, -29790, 13645, -29955, 13278, -30116, 12910
	.dc.w	-30272, 12539, -30424, 12167, -30571, 11792, -30713, 11416, -30851, 11038, -30984, 10659, -31113, 10278, -31236, 9895
	.dc.w	-31356, 9511, -31470, 9126, -31580, 8739, -31684, 8351, -31784, 7961, -31880, 7571, -31970, 7179, -32056, 6786
	.dc.w	-32137, 6392, -32213, 5997, -32284, 5601, -32350, 5205, -32412, 4807, -32468, 4409, -32520, 4011, -32567, 3611
	.dc.w	-32609, 3211, -32646, 2811, -32678, 2410, -32705, 2009, -32727, 1607, -32744, 1206, -32757, 804, -32764, 402
#endif



tbl_bitrev:		; tbl_bitrev[] = ...
#if FFT_N_512 == 512
#ifdef INPUT_IQ
	.dc.w	1*4, 257*4, 129*4, 385*4, 65*4, 321*4, 193*4, 449*4, 33*4, 289*4, 161*4, 417*4, 97*4, 353*4, 225*4, 481*4
	.dc.w	17*4, 273*4, 145*4, 401*4, 81*4, 337*4, 209*4, 465*4, 49*4, 305*4, 177*4, 433*4, 113*4, 369*4, 241*4, 497*4
	.dc.w	9*4, 265*4, 137*4, 393*4, 73*4, 329*4, 201*4, 457*4, 41*4, 297*4, 169*4, 425*4, 105*4, 361*4, 233*4, 489*4
	.dc.w	25*4, 281*4, 153*4, 409*4, 89*4, 345*4, 217*4, 473*4, 57*4, 313*4, 185*4, 441*4, 121*4, 377*4, 249*4, 505*4
	.dc.w	5*4, 261*4, 133*4, 389*4, 69*4, 325*4, 197*4, 453*4, 37*4, 293*4, 165*4, 421*4, 101*4, 357*4, 229*4, 485*4
	.dc.w	21*4, 277*4, 149*4, 405*4, 85*4, 341*4, 213*4, 469*4, 53*4, 309*4, 181*4, 437*4, 117*4, 373*4, 245*4, 501*4
	.dc.w	13*4, 269*4, 141*4, 397*4, 77*4, 333*4, 205*4, 461*4, 45*4, 301*4, 173*4, 429*4, 109*4, 365*4, 237*4, 493*4
	.dc.w	29*4, 285*4, 157*4, 413*4, 93*4, 349*4, 221*4, 477*4, 61*4, 317*4, 189*4, 445*4, 125*4, 381*4, 253*4, 509*4
	.dc.w	3*4, 259*4, 131*4, 387*4, 67*4, 323*4, 195*4, 451*4, 35*4, 291*4, 163*4, 419*4, 99*4, 355*4, 227*4, 483*4
	.dc.w	19*4, 275*4, 147*4, 403*4, 83*4, 339*4, 211*4, 467*4, 51*4, 307*4, 179*4, 435*4, 115*4, 371*4, 243*4, 499*4
	.dc.w	11*4, 267*4, 139*4, 395*4, 75*4, 331*4, 203*4, 459*4, 43*4, 299*4, 171*4, 427*4, 107*4, 363*4, 235*4, 491*4
	.dc.w	27*4, 283*4, 155*4, 411*4, 91*4, 347*4, 219*4, 475*4, 59*4, 315*4, 187*4, 443*4, 123*4, 379*4, 251*4, 507*4
	.dc.w	7*4, 263*4, 135*4, 391*4, 71*4, 327*4, 199*4, 455*4, 39*4, 295*4, 167*4, 423*4, 103*4, 359*4, 231*4, 487*4
	.dc.w	23*4, 279*4, 151*4, 407*4, 87*4, 343*4, 215*4, 471*4, 55*4, 311*4, 183*4, 439*4, 119*4, 375*4, 247*4, 503*4
	.dc.w	15*4, 271*4, 143*4, 399*4, 79*4, 335*4, 207*4, 463*4, 47*4, 303*4, 175*4, 431*4, 111*4, 367*4, 239*4, 495*4
	.dc.w	31*4, 287*4, 159*4, 415*4, 95*4, 351*4, 223*4, 479*4, 63*4, 319*4, 191*4, 447*4, 127*4, 383*4, 255*4, 511*4
#endif
	.dc.w	0*4, 256*4, 128*4, 384*4, 64*4, 320*4, 192*4, 448*4, 32*4, 288*4, 160*4, 416*4, 96*4, 352*4, 224*4, 480*4
	.dc.w	16*4, 272*4, 144*4, 400*4, 80*4, 336*4, 208*4, 464*4, 48*4, 304*4, 176*4, 432*4, 112*4, 368*4, 240*4, 496*4
	.dc.w	8*4, 264*4, 136*4, 392*4, 72*4, 328*4, 200*4, 456*4, 40*4, 296*4, 168*4, 424*4, 104*4, 360*4, 232*4, 488*4
	.dc.w	24*4, 280*4, 152*4, 408*4, 88*4, 344*4, 216*4, 472*4, 56*4, 312*4, 184*4, 440*4, 120*4, 376*4, 248*4, 504*4
	.dc.w	4*4, 260*4, 132*4, 388*4, 68*4, 324*4, 196*4, 452*4, 36*4, 292*4, 164*4, 420*4, 100*4, 356*4, 228*4, 484*4
	.dc.w	20*4, 276*4, 148*4, 404*4, 84*4, 340*4, 212*4, 468*4, 52*4, 308*4, 180*4, 436*4, 116*4, 372*4, 244*4, 500*4
	.dc.w	12*4, 268*4, 140*4, 396*4, 76*4, 332*4, 204*4, 460*4, 44*4, 300*4, 172*4, 428*4, 108*4, 364*4, 236*4, 492*4
	.dc.w	28*4, 284*4, 156*4, 412*4, 92*4, 348*4, 220*4, 476*4, 60*4, 316*4, 188*4, 444*4, 124*4, 380*4, 252*4, 508*4
	.dc.w	2*4, 258*4, 130*4, 386*4, 66*4, 322*4, 194*4, 450*4, 34*4, 290*4, 162*4, 418*4, 98*4, 354*4, 226*4, 482*4
	.dc.w	18*4, 274*4, 146*4, 402*4, 82*4, 338*4, 210*4, 466*4, 50*4, 306*4, 178*4, 434*4, 114*4, 370*4, 242*4, 498*4
	.dc.w	10*4, 266*4, 138*4, 394*4, 74*4, 330*4, 202*4, 458*4, 42*4, 298*4, 170*4, 426*4, 106*4, 362*4, 234*4, 490*4
	.dc.w	26*4, 282*4, 154*4, 410*4, 90*4, 346*4, 218*4, 474*4, 58*4, 314*4, 186*4, 442*4, 122*4, 378*4, 250*4, 506*4
	.dc.w	6*4, 262*4, 134*4, 390*4, 70*4, 326*4, 198*4, 454*4, 38*4, 294*4, 166*4, 422*4, 102*4, 358*4, 230*4, 486*4
	.dc.w	22*4, 278*4, 150*4, 406*4, 86*4, 342*4, 214*4, 470*4, 54*4, 310*4, 182*4, 438*4, 118*4, 374*4, 246*4, 502*4
	.dc.w	14*4, 270*4, 142*4, 398*4, 78*4, 334*4, 206*4, 462*4, 46*4, 302*4, 174*4, 430*4, 110*4, 366*4, 238*4, 494*4
	.dc.w	30*4, 286*4, 158*4, 414*4, 94*4, 350*4, 222*4, 478*4, 62*4, 318*4, 190*4, 446*4, 126*4, 382*4, 254*4, 510*4
#endif



;----------------------------------------------------------------------------;
#ifndef INPUT_NOISE
.global fft_input_512
.func fft_input_512
fft_input_512:
	pushw	T2H,T2L
	pushw	AH,AL
	pushw	YH,YL

	movw	XL, EL				;X = array_src;
	movw	YL, DL				;Y = array_bfly;
	clr	EH				;Zero
	ldiw	ZH,ZL, tbl_window_512		;Z = &tbl_window[0];
	ldiw	AH,AL, FFT_N_512			;A = FFT_N;
1:	lpmw	BH,BL, Z+			;B = *Z++; (window)
	ldw	CH,CL, X+			;C = *X++; (I-axis)
	FMULS16	DH,DL,T2H,T2L, BH,BL, CH,CL	;D = B * C;
	stw	Y+, DH,DL			;*Y++ = D;
#ifdef INPUT_IQ
	ldw	CH,CL, X+			;C = *X++; (Q-axis)
	FMULS16	DH,DL,T2H,T2L, BH,BL, CH,CL	;D = B * C;
#endif
	stw	Y+, DH,DL			;*Y++ = D;
	subiw	AH,AL, 1			;while(--A)
	brne	1b				;/

	popw	YH,YL
	popw	AH,AL
	popw	T2H,T2L
	clr	r1
	ret
.endfunc
#endif	/* INPUT_NOISE */



;----------------------------------------------------------------------------;
.global fft_execute_512
.func fft_execute_512
fft_execute_512:
	pushw	T2H,T2L
	pushw	T4H,T4L
	pushw	T6H,T6L
	pushw	T8H,T8L
	pushw	T10H,T10L
	pushw	T12H,T12L
	pushw	T14H,T14L
	pushw	AH,AL
	pushw	YH,YL

	movw	ZL, EL				;Z = array_bfly;
	ldiw	EH,EL, 1			;E = 1;
	ldiw	XH,XL, FFT_N_512/2			;X = FFT_N/2;
1:	ldi	AL, 4				;T12 = E; (angular speed)
	mul	EL, AL				;
	movw	T12L, T0L			;
	mul	EH, AL				;
	add	T12H, T0L			;/
	movw	T14L, EL			;T14 = E;
	pushw	EH,EL
	movw	YL, ZL				;Z = &array_bfly[0];
	mul	XL, AL				;Y = &array_bfly[X];
	addw	YH,YL, T0H,T0L			;
	mul	XH, AL				;
	add	YH, T0L				;/
	pushw	ZH,ZL
2:	clrw	T10H,T10L			;T10 = 0 (angle)
	clr	EH				;Zero reg.
3:	lddw	AH,AL, Z+0			;A = *Z - *Y; *Z++ += *Y;
	asrw	AH,AL				;
	lddw	DH,DL, Y+0			;
	asrw	DH,DL				;
	movw	CL, AL				;
	subw	AH,AL, DH,DL			;
	addw	CH,CL, DH,DL			;
	stw	Z+, CH,CL			;/
	lddw	BH,BL, Z+0			;B = *Z - *Y; *Z++ += *Y;
	asrw	BH,BL				;
	lddw	DH,DL, Y+2			;
	asrw	DH,DL				;
	movw	CL, BL				;
	subw	BH,BL, DH,DL			;
	addw	CH,CL, DH,DL			;
	stw	Z+, CH,CL			;/
	movw	T0L, ZL
	ldiw	ZH,ZL, tbl_cos_sin		;C = cos(T10); D = sin(T10);
	addw	ZH,ZL, T10H,T10L		;
	lpmw	CH,CL, Z+			;
	lpmw	DH,DL, Z+			;/
	movw	ZL, T0L
	FMULS16	T4H,T4L,T2H,T2L, AH,AL, CH,CL	;*Y++ = A * C + B * D;
	FMULS16	T8H,T8L,T6H,T6L, BH,BL, DH,DL	;
	addd	T4H,T4L,T2H,T2L, T8H,T8L,T6H,T6L;
	stw	Y+, T4H,T4L			;/
	FMULS16	T4H,T4L,T2H,T2L, BH,BL, CH,CL 	;*Y++ = B * C - A * D;
	FMULS16	T8H,T8L,T6H,T6L, AH,AL, DH,DL 	;
	subd	T4H,T4L,T2H,T2L, T8H,T8L,T6H,T6L;
	stw	Y+, T4H,T4L			;/
	addw	T10H,T10L, T12H,T12L		;T10 += T12; (next angle)
#if FFT_N_512 >= 128
	sbrs	T10H, FFT_B_512 - 7			;while(T10 < pi)
#else
	sbrs	T10L, FFT_B_512 + 1
#endif
	rjmp	3b				;/
	ldi	AL, 4				;Y += X; Z += X; (skip split segment)
	mul	XL, AL
	addw	YH,YL, T0H,T0L			;
	addw	ZH,ZL, T0H,T0L			;
	mul	XH, AL				;
	add	YH, T0L				;
	add	ZH, T0L				;/
	ldi	EL, 1				;while(--T14)
	subw	T14H,T14L, EH,EL		;
	rjne	2b				;/
	popw	ZH,ZL
	popw	EH,EL
	lslw	EH,EL				;E *= 2;
	lsrw	XH,XL				;while(X /= 2)
	adiw	XL, 0				;
	rjne	1b				;/

	popw	YH,YL
	popw	AH,AL
	popw	T14H,T14L
	popw	T12H,T12L
	popw	T10H,T10L
	popw	T8H,T8L
	popw	T6H,T6L
	popw	T4H,T4L
	popw	T2H,T2L
;	clr	r1
	ret
.endfunc



;----------------------------------------------------------------------------;
.global fft_output_512
.func fft_output_512
fft_output_512:
	pushw	T2H,T2L
	pushw	T4H,T4L
	pushw	T6H,T6L
	pushw	T8H,T8L
	pushw	T10H,T10L
	pushw	AH,AL
	pushw	YH,YL

	movw	T10L, EL			;T10 = array_bfly;
	movw	YL, DL				;Y = array_output;
	ldiw	ZH,ZL, tbl_bitrev		;Z = tbl_bitrev;
	clr	EH				;Zero
#ifdef INPUT_IQ
	ldiw	AH,AL, FFT_N_512			;A = FFT_N; (plus/minus)
#else
	ldiw	AH,AL, FFT_N_512 / 2		;A = FFT_N / 2; (plus only)
#endif
1:	lpmw	XH,XL, Z+			;X = *Z++;
	addw	XH,XL, T10H,T10L		;X += array_bfly;
	ldw	BH,BL, X+			;B = *X++;
	ldw	CH,CL, X+			;C = *X++;
	FMULS16	T4H,T4L,T2H,T2L, BH,BL, BH,BL	;T4:T2 = B * B;
	FMULS16	T8H,T8L,T6H,T6L, CH,CL, CH,CL	;T8:T6 = C * C;
	addd	T4H,T4L,T2H,T2L, T8H,T8L,T6H,T6L;T4:T2 += T8:T6;
	SQRT32					;B = sqrt(T4:T2);
	stw	Y+, BH,BL			;*Y++ = B;
	subiw	AH,AL, 1			;while(--A)
	rjne	1b				;/

	popw	YH,YL
	popw	AH,AL
	popw	T10H,T10L
	popw	T8H,T8L
	popw	T6H,T6L
	popw	T4H,T4L
	popw	T2H,T2L
	clr	r1
	ret
.endfunc



;----------------------------------------------------------------------------;
.global fmuls_f_512
.func fmuls_f_512
fmuls_f_512:
	movw	CL, EL				;C = E;
	clr	EH	;Zero
	FMULS16	ZH,ZL,XH,XL, CH,CL, DH,DL	;Z:X = C * D;
	movw	EL, ZL
	clr	r1
	ret
.endfunc

